<?php

namespace App\Http\Controllers;
use App\Models\Product;
use App\Models\Bill;
use App\Models\BillItem;
use App\Models\CreditDetail;
use App\Models\PrintSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage; 
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;


class ProductController extends Controller
{
    public function return()
   {
    return view('product.return');
   }


public function lowSalesFilter(Request $request)
{
    $products = collect(); // Default empty collection
    
    // Get filter parameters
    $quickFilter = $request->get('quick_filter');
    $customDate = $request->get('since_date');

    // If any filter is applied
    if ($quickFilter || $customDate) {
        $sinceDate = null;

        // Determine the date based on quick filter or custom date
        if ($quickFilter && $quickFilter !== 'custom') {
            $sinceDate = Carbon::now()->subDays($quickFilter)->toDateString();
        } elseif ($customDate) {
            $sinceDate = $customDate;
        }

        // Query products with no sales since the determined date
        if ($sinceDate) {
            $products = Product::whereDoesntHave('billitems.bill', function($query) use ($sinceDate) {
                $query->where('date', '>=', $sinceDate);
            })
            ->get();
        }
    }

    return view('product.lowSale', compact('products'));
}
public function searchStarts(Request $request)
{
    $query = $request->get('q');

    $products = Product::where('name', 'LIKE', "{$query}%")
        ->orWhere('company', 'LIKE', "{$query}%")
        ->limit(10)
        ->get(['id', 'name', 'company']);

    return response()->json($products);
}
public function printSelected(Request $request)
{
     $printSetting = PrintSetting::first();
    $ids = $request->input('ids'); // already an array
    $products = Product::whereIn('id', $ids)->get();

    return view('product.lowSalePrint', compact('products','printSetting'));
}

public function index(Request $request, $filter = 'all')
{
    $currentDate = now();
    $oneMonthLater = $currentDate->copy()->addMonth();
    $search = $request->input('search');

    $query = Product::query();

    // Apply filter
    switch ($filter) {
        case 'in-stock':
            $query->where('quantity', '>', 0);
            break;

        case 'out-of-stock':
            $query->where('quantity', '=', 0);
            break;

        case 'low-stock':
            $query->where('quantity', '>', 0)->where('quantity', '<=', 10);
            break;

        case 'near-expiry':
            $query->where('quantity', '>', 0)
                  ->whereBetween('expiry_date', [$currentDate, $oneMonthLater]);
            break;

        case 'expired':
            $query->where('expiry_date', '<', $currentDate);
            break;

        case 'all':
        default:
            // No additional condition
            break;
    }

    // Apply search
    if (!empty($search)) {
        $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('company', 'like', "%{$search}%");
        });
    }

    $products = $query->get();

    // Manually add expiry status flags
    $products->each(function ($product) use ($currentDate, $oneMonthLater) {
        // Set expiry status flags (not saved to database)
        $product->is_expired = $product->expiry_date < $currentDate;
        $product->is_near_expiry = !$product->is_expired && 
                                  ($product->expiry_date >= $currentDate && 
                                   $product->expiry_date <= $oneMonthLater);
        
        // Maintain existing stock flags
        $product->is_low_stock = $product->quantity <= $product->low_stock_alert;
        $product->out_stock = $product->quantity == 0;
    });

    return view('product.index', compact('products', 'filter', 'search'));
}


   
   public function productAdd()
   {
    return view('product.create');
   }

   public function productStore(Request $request)
    {
       //dd($request->all());
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'salt' => 'nullable|string|max:255',
            'company' => 'nullable|string|max:255',
            'categories' => 'required|array',
            'batch_number' => 'required|string|max:255',
            'quantity' => 'required|integer',
           'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'expiry_date' => 'required|date',
            'expiry_alert' => 'nullable|integer',
            'low_stock_alert' => 'nullable|integer',
            'purchase_from' => 'nullable|string|max:255',
            'retail_price' => 'nullable|numeric',
            'purchase_price' => 'nullable|numeric',
            'local_price' => 'nullable|numeric',
            'doctor_price' => 'nullable|numeric',
            'unit_type' => 'required|string',
            'pieces_per_packet' => 'nullable|integer',
            'price_per_piece' => 'nullable|numeric',
        ]);
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('products', 'public');
        }
        Product::create([
            'name' => $validated['name'],
            'type' => $validated['type'],
            'salt' => $validated['salt'] ?? null,
            'company' => $validated['company'] ?? null,
            'categories' => $validated['categories'],
            'batch_number' => $validated['batch_number'],
            'quantity' => $validated['quantity'],
            'image' =>$imagePath,
            'expiry_date' => $validated['expiry_date'],
            'expiry_alert' => $validated['expiry_alert'] ?? 30,
            'low_stock_alert' => $validated['low_stock_alert'] ?? 10,
            'purchase_from' => $validated['purchase_from'] ?? null,
            'retail_price' => $validated['retail_price'] ?? null,
            'purchase_price' => $validated['purchase_price'] ?? null,
            'local_price' => $validated['local_price'] ?? null,
            'doctor_price' => $validated['doctor_price'] ?? null,
            'unit_type' => $validated['unit_type'],
            'pieces_per_packet' => $validated['pieces_per_packet'] ?? null,
            'price_per_piece' => $validated['price_per_piece'] ?? null,
        ]);

        return redirect()->route('product.index')->with('success', 'Product added successfully.');
    }
    public function edit($id)
    {
        $product = Product::findOrFail($id);
        return view('product.edit', compact('product'));
    }
    
   public function update(Request $request, $id)
{
    $product = Product::findOrFail($id);

    $request->validate([
        'name' => 'required|string|max:255',
        'type' => 'required|string|max:255',
        'salt' => 'nullable|string|max:255',
        'company' => 'nullable|string|max:255',
        'categories' => 'required|array',
        'batch_number' => 'required|string|max:255',
        'quantity' => 'required|integer',
        'expiry_date' => 'required|date',
        'expiry_alert' => 'nullable|integer',
        'low_stock_alert' => 'nullable|integer',
        'purchase_from' => 'nullable|string|max:255',
        'retail_price' => 'nullable|numeric',
        'purchase_price' => 'nullable|numeric',
        'local_price' => 'nullable|numeric',
        'doctor_price' => 'nullable|numeric',
        'unit_type' => 'required|string',
        'pieces_per_packet' => 'nullable|integer',
        'price_per_piece' => 'nullable|numeric',
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // 2MB max
    ]);

    $data = [
        'name' => $request->name,
        'type' => $request->type,
        'salt' => $request->salt,
        'company' => $request->company,
        'categories' => $request->categories,
        'batch_number' => $request->batch_number,
        'quantity' => $request->quantity,
        'expiry_date' => $request->expiry_date,
        'expiry_alert' => $request->expiry_alert,
        'low_stock_alert' => $request->low_stock_alert,
        'purchase_from' => $request->purchase_from,
        'retail_price' => $request->retail_price,
        'purchase_price' => $request->purchase_price,
        'local_price' => $request->local_price,
        'doctor_price' => $request->doctor_price,
        'unit_type' => $request->unit_type,
        'pieces_per_packet' => $request->pieces_per_packet,
        'price_per_piece' => $request->price_per_piece,
    ];

    // Handle image removal if checkbox is checked
    if ($request->has('remove_image')) {
        // Delete old image if exists
        if ($product->image) {
            Storage::delete('public/' . $product->image);
            $data['image'] = null;
        }
    }
    // Handle new image upload
    elseif ($request->hasFile('image')) {
        // Delete old image if exists
        if ($product->image) {
            Storage::delete('public/' . $product->image);
        }
        
        // Store new image
        $imagePath = $request->file('image')->store('products', 'public');
        $data['image'] = $imagePath;
    }
    // If no new image and no removal, keep existing image (no action needed)

    $product->update($data);

    return redirect()->route('product.index')->with('success', 'Product updated successfully.');
}

     public function searchBillItems(Request $request)
{
    $query = $request->input('q');

    $billItems = BillItem::with(['product', 'bill.customer'])
        ->where(function($q) use ($query) {
            $q->whereHas('product', function($q) use ($query) {
                $q->where('name', 'like', "%$query%");
            })
            ->orWhereHas('bill', function($q) use ($query) {
                 $q->where('bill_id', 'like', "%$query%")
                  ->orWhereHas('customer', function($q2) use ($query) {
                      $q2->where('name', 'like', "%$query%")
                         ->orWhere('phone', 'like', "%$query%");
                  });
            });
        })
        ->where('qty', '>', 0)
        ->get()
        ->each(function ($item) {
            $item->returnable_qty = $item->qty;
        });

    return view('product.return', [
        'billItems' => $billItems,
        'query' => $query
    ]);
}

    // Handle the submission of returned products
 public function submitCart(Request $request)
{
    logger()->debug('Incoming return data:', $request->all());
    
    $validated = $request->validate([
        'products' => 'required|array|min:1',
        'products.*.product_id' => 'required|exists:products,id',
        'products.*.qty' => 'required|integer|min:1',
        'products.*.bill_item_id' => 'required|exists:bill_items,id',
    ]);

    DB::beginTransaction();
    try {
        $processedItems = 0;
        
        foreach ($validated['products'] as $product) {
            $productRecord = Product::findOrFail($product['product_id']);
            $billItem = BillItem::with(['bill', 'bill.customer'])->findOrFail($product['bill_item_id']);
            
            if (!$billItem->bill) {
                throw new \Exception("Associated bill not found for bill item {$billItem->id}");
            }

            $returnQty = min($product['qty'], $billItem->qty);
            if ($returnQty <= 0) continue;

            $unitPrice = $billItem->price; // Using 'amount' instead of 'price'
            $totalReturnAmount = $returnQty * $unitPrice;

            // 1. Update product stock
            $productRecord->increment('quantity', $returnQty);

            // 2. Update bill item
            $billItem->decrement('qty', $returnQty);
             $billItem->decrement('amount', $totalReturnAmount);
             $billItem->decrement('price', $unitPrice);
            // No need to update total since we're using amount*quantity
            $billItem->save();

            // 3. Update bill totals
            $bill = $billItem->bill;
            $bill->decrement('total_amount', $totalReturnAmount);
            
            // 4. Handle credit payments
            // 4. Handle credit payments
if ($bill->payment === 'credit') {
    $credit = CreditDetail::where('customer_id', $bill->customer_id)->first();

    if ($credit) {
        $credit->due_amount = max(0, $credit->due_amount - $totalReturnAmount);
        $credit->save();
    }
}


            
            $bill->save();
            $processedItems++;
        }

        DB::commit();
        
         if ($request->wantsJson()) {
        return response()->json([
            'success' => true,
            'message' => "$processedItems product(s) returned successfully",
            'redirect' => route('product.index')
        ]);
    }

    // Regular form submission
    return redirect()->route('product.index')
           ->with('success', "$processedItems product(s) returned successfully");

} catch (\Exception $e) {
    DB::rollBack();
    logger()->error('Return processing failed: ' . $e->getMessage());

    if ($request->wantsJson()) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to process return: ' . $e->getMessage()
        ], 500);
    }
    return back()->with('error', 'Failed to process return: ' . $e->getMessage());
}
}
    
    
    public function delete($id)
{
    $product = Product::findOrFail($id);
    $product->delete();

    return redirect()->route('product.index')->with('success', 'Product deleted successfully.');
}

public function search(Request $request)
{
    $query = $request->input('q');
    $view = $request->input('view', 'default.searchResults'); // fallback view

    $products = Product::where('name', 'like', "%$query%")
        ->orWhere('type', 'like', "%$query%")
        ->orWhere('company', 'like', "%$query%")
        ->orWhere('categories', 'like', "%$query%")
        ->orWhere('salt', 'like', "%$query%")
        ->get();

    return view($view, compact('products', 'query'));
}
public function todaySales(Request $request)
{
    $query = BillItem::join('bills', 'bill_items.bill_id', '=', 'bills.id')
        ->join('products', 'bill_items.product_id', '=', 'products.id');
    
    // Apply date filter if provided
    if ($request->has('start_date') && $request->has('end_date')) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        $query->whereBetween('bills.date', [$startDate, $endDate]);
    } else {
        // Default to today if no dates provided
        $query->whereDate('bills.date', Carbon::today());
    }
    
    $sales = $query->groupBy('bill_items.product_id', 'products.name', 'products.type', 'bills.date')
        ->select(
            'products.name as product_name',
            'products.type as product_type',
            DB::raw('SUM(bill_items.qty) as sold_qty'),
            DB::raw('SUM(bill_items.amount) as sold_amount'),
            DB::raw('DATE(bills.date) as sale_date')
        )
        ->get();

    return view('product.sale', compact('sales'));
}

public function salesReport(Request $request)
{
    $cashQuery = Bill::with('billitems.product')
    ->where('payment', 'cash');

$creditQuery = Bill::with(['billitems.product', 'customer'])
    ->where('payment', 'credit');

    // Apply date filter if provided
    if ($request->has('start_date') && $request->has('end_date')) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        $cashQuery->whereBetween('date', [$startDate, $endDate]);
        $creditQuery->whereBetween('date', [$startDate, $endDate]);
    } else {
        // Default to today if no dates provided
        $cashQuery->whereDate('date', Carbon::today());
        $creditQuery->whereDate('date', Carbon::today());
    }

    // Get formatted sales data
    $cashSales = $cashQuery->get()
        ->flatMap(function ($bill) {
            return $bill->billitems->map(function ($item) use ($bill) {
                return [
                    'bill_id' => $bill->id,
                    'product_name' => $item->product->name,
                    'product_type' => $item->product->type,
                    'sold_qty' => $item->qty,
                    'sold_amount' => $item->amount,
                    'sale_date' => Carbon::parse($bill->date)->format('Y-m-d'), // Format here
                ];
            });
        });

    $creditSales = $creditQuery->get()
        ->flatMap(function ($bill) {
            return $bill->billitems->map(function ($item) use ($bill) {
                return [
                    'bill_id' => $bill->id,
                    'customer_name' => $bill->customer->name ?? 'N/A',
                    'product_name' => $item->product->name,
                    'product_type' => $item->product->type,
                    'sold_qty' => $item->qty,
                    'sold_amount' => $item->amount,
                    'sale_date' => Carbon::parse($bill->date)->format('Y-m-d'), // Format here
                ];
            });
        });

    return view('product.cash-in', compact('cashSales', 'creditSales'));
}
public function productSales(Product $product)
{
    $sales = BillItem::with(['bill.customer', 'bill'])
        ->where('product_id', $product->id)
        ->orderBy('created_at', 'desc')
        ->paginate(10);

    $totalQuantity = BillItem::where('product_id', $product->id)->sum('qty');
    $totalAmount = BillItem::where('product_id', $product->id)->sum('amount');

    
    return view('product.productSale', compact('sales', 'totalQuantity', 'totalAmount', 'product'));
}

public function print()
{
     $printSetting = PrintSetting::first();
    $products = Product::all(); // eager load categories if it's a relation

    return view('product.print', compact('products','printSetting'));
}
public function makebillsearchStarts(Request $request)
{
    $query = $request->get('q');

    $products = Product::where('name', 'LIKE', "{$query}%")
        ->orWhere('company', 'LIKE', "{$query}%")
        ->limit(10)
        ->get(['id', 'name', 'company']);

    return response()->json($products);
}
}
