@extends('layouts.app')

@section('content')
<div class="max-w-6xl mx-auto px-4 py-6">

    <h1 class="text-2xl font-bold mb-4 text-center">Search Product for Bill</h1>

    {{-- Buy Outside Button --}}
    <div class="mb-4 flex justify-end">
        <button id="buyOutsideBtn" 
                class="bg-purple-600 text-white px-4 py-2 rounded hover:bg-purple-700 transition-colors">
            Buy Outside
        </button>
    </div>

    {{-- Search Form --}}
    <form method="GET" action="{{ route('products.search') }}" class="mb-6 w-full">
    <div class="flex flex-col sm:flex-row gap-2 w-full">
        <!-- Parent container for input + results (takes full width) -->
        <div class="relative w-full">
            <!-- Search input (full width) -->
            <input 
                type="text" 
                name="q" 
                value="{{ old('q', $query ?? '') }}"  
                id="liveSearchInput" 
                required
                placeholder="Search by name, salt, category, company, type..."
                class="w-full border border-gray-300 rounded p-2"
                oninput="handleLiveSearch(this.value)"
                autocomplete="off" 
            />
            <!-- Live results container (full width + positioned below input) -->
            <div id="liveResultContainer"
     class="absolute top-full left-0 w-full bg-white border border-gray-300 rounded shadow text-sm z-10 hidden"
     style="max-height: 300px; overflow-y: auto;">
</div>

        </div>
        <!-- Hidden field and submit button -->
        <input type="hidden" name="view" value="bill.makeBill">
        <button 
            type="submit"
            class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 whitespace-nowrap"
        >
            Search
        </button>
    </div>
</form>
    {{-- Modal for Product Quantity --}}
    <div id="productQuantityModal" class="fixed z-50 inset-0 overflow-y-auto hidden">
        <div class="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div class="fixed inset-0 transition-opacity" aria-hidden="true">
                <div class="absolute inset-0 bg-gray-500 opacity-75"></div>
            </div>
            <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
            <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                    <div class="sm:flex sm:items-start">
                        <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                            <div class="flex justify-between items-center">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">Buy Outside Products</h3>
                                <button id="closeModal" class="text-gray-400 hover:text-gray-500">
                                    <span class="sr-only">Close</span>
                                    <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                                    </svg>
                                </button>
                            </div>
                            <div class="mt-4">
                                <label for="productQuantity" class="block text-sm font-medium text-gray-700">Number of Products:</label>
                                <input type="number" id="productQuantity" min="1" value="1" 
                                       class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                    <button id="generateFormBtn" type="button" class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-indigo-600 text-base font-medium text-white hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:ml-3 sm:w-auto sm:text-sm">
                        Generate Form
                    </button>
                    <button id="cancelModal" type="button" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    {{-- Outside Order Form --}}
    <div id="outsideOrderFormContainer" class="hidden mb-6">
        <form id="outsideOrderForm" method="POST" action="{{route('orders.storeOutside')}}">
            @csrf
            <div class="bg-white shadow-md rounded-lg p-6">
                <h2 class="text-xl font-semibold mb-4">Outside Purchase</h2>
                <div id="productsContainer" class="space-y-4"></div>
                
                <div class="mt-6 pt-4 border-t">
                    <h3 class="text-lg font-medium mb-3">Customer Information</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="customer_name" class="block text-sm font-medium text-gray-700">Customer Name:</label>
                            <input type="text" id="customer_name" name="customer_name" required
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="payment_type" class="block text-sm font-medium text-gray-700">Payment Type:</label>
                            <select id="payment_type" name="payment_type" required
                                    class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                                <option value="cash">Cash</option>
                                <option value="card">Card</option>
                                <option value="online">Online Payment</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mt-4">
                        <label for="total_amount" class="block text-sm font-medium text-gray-700">Total Amount:</label>
                        <input type="number" id="total_amount" name="total_amount" readonly
                               class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 bg-gray-100 sm:text-sm">
                    </div>
                    
                    <input type="hidden" name="sold_as" value="outside">
                </div>
                
                <div class="mt-6 flex justify-end space-x-3">
                    <button type="button" id="cancelOutsideOrder" class="bg-gray-300 text-gray-700 px-4 py-2 rounded hover:bg-gray-400">
                        Cancel
                    </button>
                    <button type="submit" class="bg-indigo-600 text-white px-4 py-2 rounded hover:bg-indigo-700">
                        Submit Order
                    </button>
                </div>
            </div>
        </form>
    </div>

    {{-- Rest of your existing content (Search Results and Cart Section) --}}
    @if(isset($products) && $products->count())
    <h2 class="text-lg font-semibold mb-3">Search Results</h2>
    
    <div class="grid gap-4 sm:grid-cols-2 lg:grid-cols-3 mb-6">
       @foreach($products as $product)
    <div class="border p-4 rounded-lg shadow-md bg-white flex flex-col justify-between h-full relative min-h-[320px]">
        {{-- Top Section: Details and Price --}}
        <div class="flex justify-between gap-4 mb-4">
            {{-- Left: Product Details --}}
            <div class="flex-1">
                <p class="font-bold text-lg mb-1">{{ $product->name }}</p>
                <p class="text-sm text-gray-500 mb-1">Company: {{ $product->company }}</p>
                <p class="text-sm text-gray-500 mb-1">
                    Category:
                    @foreach ((is_array($product->categories) ? $product->categories : json_decode($product->categories, true) ?? []) as $category)
                        <span class="inline-block bg-gray-200 text-gray-700 px-2 py-1 rounded mr-1 mb-1">{{ $category }}</span>
                    @endforeach
                </p>
                <p class="text-sm text-gray-500 mb-1">Salt: {{ $product->salt }}</p>
                <p class="text-sm text-gray-500 mb-1">Unit Type: {{ $product->unit_type }}</p>
                <p class="text-sm text-gray-500">Stock: {{ $product->quantity }} left</p>
            </div>

            {{-- Right: Price Details --}}
            <div class="text-right text-sm text-gray-600 min-w-[130px] space-y-1">
                <p><span class="font-semibold">Retail:</span> {{ $product->retail_price }}</p>
                <p><span class="font-semibold">Doctor:</span> {{ $product->doctor_price }}</p>
                <p><span class="font-semibold">Local:</span> {{ $product->local_price }}</p>
                <p><span class="font-semibold">Purchase:</span> {{ $product->purchase_price }}</p>
            </div>
        </div>

        {{-- Product Image --}}
        <div class="flex-grow flex items-center justify-center mb-4">
            <div class="w-full h-32 border overflow-hidden rounded-lg shadow-sm">
                <img src="{{ asset('storage/' . $product->image) }}"
                     alt="{{ $product->name }}"
                     class="object-contain w-full h-full">
            </div>
        </div>

        {{-- Bottom Section: Button --}}
        <div class="mt-auto pt-4 text-center border-t">
            @php
                $productJson = json_encode([
                    'id' => $product->id,
                    'name' => $product->name,
                    'company' => $product->company,
                    'unit_type' => $product->unit_type,
                    'salt' => $product->salt,
                    'stock' => $product->quantity,
                    'category' => is_array($product->categories) ? implode(', ', $product->categories) : $product->categories,
                ]);
            @endphp

            @if ($product->quantity > 0)
                <button type="button"
                    onclick='addToCart({!! $productJson !!})'
                    class="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors w-full">
                    Add to Bill
                </button>
            @else
                <div class="text-red-600 font-semibold mb-2">
                    Out of Stock
                </div>
                <a href="{{ route('buyOutside', $product->id) }}"
                   class="inline-block bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors w-full">
                    Buy Outside
                </a>
            @endif
        </div>
    </div>
@endforeach
    </div>
@endif

    {{-- Cart Section --}}
    <h2 class="text-xl font-semibold mb-3">Cart</h2>
    <form method="POST" action="{{ route('cart.submit') }}" id="bill-form">
        @csrf
        <div id="cart-container" class="space-y-4"></div>

        <button type="submit"
            class="mt-4 bg-indigo-600 text-white px-6 py-2 rounded hover:bg-indigo-700 hidden"
            id="submit-btn">
            Submit Bill
        </button>
    </form>

</div>
@endsection

@section('scripts')
<script>
    let cart = JSON.parse(localStorage.getItem('cart')) || [];

    function saveCart() {
        localStorage.setItem('cart', JSON.stringify(cart));
    }

    function addToCart(product) {
        const exists = cart.find(p => p.id === product.id);
        if (exists) {
            alert('Product already in cart!');
            return;
        }

        cart.push(product);
        saveCart();
        renderCart();
    }

    function removeFromCart(productId) {
        cart = cart.filter(p => p.id !== productId);
        saveCart();
        renderCart();
    }

    function renderCart() {
        const container = document.getElementById('cart-container');
        container.innerHTML = '';
        if (cart.length === 0) {
            document.getElementById('submit-btn').classList.add('hidden');
            return;
        }
        document.getElementById('submit-btn').classList.remove('hidden');

        document.getElementById('bill-form')?.addEventListener('submit', function () {
            localStorage.removeItem('cart');
        });

        cart.forEach((product, index) => {
            const item = document.createElement('div');
            item.className = 'p-4 border rounded bg-gray-50 shadow flex flex-col sm:flex-row justify-between items-center gap-2';
            item.innerHTML = `
                <div>
                    <input type="hidden" name="products[${index}][product_id]" value="${product.id}">
                    <p class="font-bold">${product.name}</p>
                    <p class="text-sm text-gray-500">${product.company} | ${product.category}</p>
                    <p class="text-sm text-gray-500">Salt: ${product.salt} | Unit: ${product.unit_type}</p>
                </div>
                <div class="flex items-center gap-2">
                    <label class="text-sm">Qty:</label>
                    <input type="number" name="products[${index}][qty]" value="1" min="1" class="w-20 p-1 border rounded">
                    <button type="button" onclick="removeFromCart(${product.id})"
                        class="text-red-600 text-sm hover:underline">Remove</button>
                </div>
            `;
            container.appendChild(item);
        });
    }

    // Outside Purchase Functionality
    document.addEventListener("DOMContentLoaded", function() {
        renderCart();
        
        // Get elements for outside purchase
        const buyOutsideBtn = document.getElementById('buyOutsideBtn');
        const productQuantityModal = document.getElementById('productQuantityModal');
        const closeModal = document.getElementById('closeModal');
        const cancelModal = document.getElementById('cancelModal');
        const generateFormBtn = document.getElementById('generateFormBtn');
        const productQuantityInput = document.getElementById('productQuantity');
        const outsideOrderFormContainer = document.getElementById('outsideOrderFormContainer');
        const cancelOutsideOrder = document.getElementById('cancelOutsideOrder');
        const productsContainer = document.getElementById('productsContainer');
        const totalAmountInput = document.getElementById('total_amount');
        
        // Open modal when Buy Outside is clicked
        buyOutsideBtn.addEventListener('click', function() {
            productQuantityModal.classList.remove('hidden');
        });
        
        // Close modal
        function closeQuantityModal() {
            productQuantityModal.classList.add('hidden');
        }
        
        closeModal.addEventListener('click', closeQuantityModal);
        cancelModal.addEventListener('click', closeQuantityModal);
        
        // Generate form when quantity is entered
        generateFormBtn.addEventListener('click', function() {
            const productCount = parseInt(productQuantityInput.value);
            
            if (productCount > 0) {
                generateProductFields(productCount);
                closeQuantityModal();
                outsideOrderFormContainer.classList.remove('hidden');
            }
        });
        
        // Cancel outside order
        cancelOutsideOrder.addEventListener('click', function() {
            outsideOrderFormContainer.classList.add('hidden');
        });
        
        // Function to generate product fields
        function generateProductFields(count) {
            productsContainer.innerHTML = '';
            
            for (let i = 0; i < count; i++) {
                const productDiv = document.createElement('div');
                productDiv.className = 'border p-4 rounded-lg bg-gray-50';
                productDiv.innerHTML = `
                    <h4 class="font-medium mb-3">Product ${i + 1}</h4>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="product_name_${i}" class="block text-sm font-medium text-gray-700">Name:</label>
                            <input type="text" id="product_name_${i}" name="products[${i}][name]" required
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="product_type_${i}" class="block text-sm font-medium text-gray-700">Type:</label>
                            <input type="text" id="product_type_${i}" name="products[${i}][type]" required
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="product_qty_${i}" class="block text-sm font-medium text-gray-700">Quantity:</label>
                            <input type="number" id="product_qty_${i}" name="products[${i}][qty]" min="1" value="1" required
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm qty-input">
                        </div>
                        <div>
                            <label for="product_price_${i}" class="block text-sm font-medium text-gray-700">Price:</label>
                            <input type="number" id="product_price_${i}" name="products[${i}][price]" min="0" step="0.01" required
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm price-input">
                        </div>
                    </div>
                `;
                productsContainer.appendChild(productDiv);
            }
            
            // Add event listeners for dynamic total calculation
            document.querySelectorAll('.qty-input, .price-input').forEach(input => {
                input.addEventListener('input', calculateTotal);
            });
            
            // Initial calculation
            calculateTotal();
        }
        
        // Calculate total amount
        function calculateTotal() {
            let total = 0;
            const productCount = parseInt(productQuantityInput.value);
            
            for (let i = 0; i < productCount; i++) {
                const qtyInput = document.getElementById(`product_qty_${i}`);
                const priceInput = document.getElementById(`product_price_${i}`);
                
                if (qtyInput && priceInput) {
                    const qty = parseFloat(qtyInput.value) || 0;
                    const price = parseFloat(priceInput.value) || 0;
                    total += qty * price;
                }
            }
            
            totalAmountInput.value = total.toFixed(2);
        }
    });


 let selectedIndex = -1;
let currentSuggestions = [];

function handleLiveSearch(query) {
    const container = document.getElementById('liveResultContainer');

    if (!query.trim()) {
        container.classList.add('hidden');
        container.innerHTML = '';
        currentSuggestions = [];
        selectedIndex = -1;
        return;
    }

    fetch(`{{ route('bill.searchStarts') }}?q=${encodeURIComponent(query)}`, {
        headers: { 'Accept': 'application/json' }
    })
    .then(res => res.json())
    .then(data => {
        currentSuggestions = data;
        selectedIndex = -1;

        if (data.length === 0) {
            container.innerHTML = `<div class="px-3 py-2 text-gray-500">No matching products</div>`;
        } else {
            container.innerHTML = data.map((product, index) => `
                <div class="px-3 py-2 suggestion-item cursor-pointer border-b border-gray-200"
                     data-index="${index}"
                     onclick="selectSuggestion('${product.name.replace(/'/g, "\\'")}')">
                    <strong>${product.name}</strong> <span class="text-gray-500">(${product.company})</span>
                </div>
            `).join('');
        }
        container.classList.remove('hidden');
    })
    .catch(err => {
        console.error(err);
        container.classList.add('hidden');
    });
}

function selectSuggestion(name) {
    const input = document.getElementById('liveSearchInput');
    input.value = name;

    const container = document.getElementById('liveResultContainer');
    container.classList.add('hidden');
    container.innerHTML = '';

    input.focus();
    // Optional: auto-submit
    // input.form.submit();
}

function updateActiveItem(items) {
    items.forEach((item, index) => {
        if (index === selectedIndex) {
            item.classList.add('bg-blue-100');
            item.scrollIntoView({ block: 'nearest' });
        } else {
            item.classList.remove('bg-blue-100');
        }
    });
}

// Keyboard navigation
document.addEventListener('DOMContentLoaded', function () {
    const input = document.getElementById('liveSearchInput');

    input.addEventListener('keydown', function (e) {
        const container = document.getElementById('liveResultContainer');
        const items = container.querySelectorAll('[data-index]');

        if (container.classList.contains('hidden') || items.length === 0) return;

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            selectedIndex = (selectedIndex + 1) % items.length;
            document.body.classList.add('using-keyboard');
            updateActiveItem(items);
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            selectedIndex = (selectedIndex - 1 + items.length) % items.length;
            document.body.classList.add('using-keyboard');
            updateActiveItem(items);
        } else if (e.key === 'Enter' && selectedIndex >= 0) {
            e.preventDefault();
            const selectedProduct = currentSuggestions[selectedIndex];
            selectSuggestion(selectedProduct.name);
        }
    });
});

// Hide suggestions if clicked outside
document.addEventListener('click', function (e) {
    const input = document.getElementById('liveSearchInput');
    const container = document.getElementById('liveResultContainer');
    if (!input.contains(e.target) && !container.contains(e.target)) {
        container.classList.add('hidden');
    }
});

// Remove hover highlight while using keyboard
let usingKeyboard = false;
document.addEventListener('keydown', (e) => {
    if (['ArrowUp', 'ArrowDown'].includes(e.key)) {
        usingKeyboard = true;
        document.body.classList.add('using-keyboard');
    }
});
document.addEventListener('mousemove', () => {
    if (usingKeyboard) {
        usingKeyboard = false;
        document.body.classList.remove('using-keyboard');
    }
});
</script>
<style>
/* Prevent hover override when navigating with keyboard */
body.using-keyboard #liveResultContainer .suggestion-item:hover {
    background-color: inherit !important;
}
</style>
@endsection